/*********************************************************************
*    Project: MG82F6D16-DEMO
*            MG82F6Dxx	SSOP28 EV Board (TH189A)
*			CpuCLK=24MHz, SysCLK=48MHz
*    Description:
*			SPI(nSS/P14,MOSI/P15,MISO/P16,SPICLK/P17) Read/Write Flash
*			SPI clock :12MHz
*    Note:
* 
*    Creat time:
*    Modify:
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F6D16.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6D16.H"
#include ".\include\API_Uart_BRGRL_MG82F6D16.H"


/*************************************************
*Set SysClk (MAX.50MHz)
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		48000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.25MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
//#define MCU_CPUCLK		(MCU_SYSCLK)
#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define TIMER_12T_10ms_TH	((65536-(u16)(float)(10000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_10ms_TL	((65536-(u16)(float)(10000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)


#define LED_G_0		P22
#define LED_R		P24
#define LED_G_1		P26

#define SPI_nSS		P14


// SPI
#define FLASH_ID					0x00C22016L					// MX25L3206

#define Dummy_Data					0xFF
#define FLASH_BLOCK_SIZE			(64L*1024L)					// 64K Byte

#define FLASH_TEST_START_ADDRESS	0x00000000
#define FLASH_TEST_SIZE_MAX			(1024L*1024L)				// 1M Byte

#define TEST_BUF_SIZE				512
xdata u8 TestBuf[TEST_BUF_SIZE];
u16 TestBufCnt;

u8	SPIOvTime;

/***********************************************************************************
*Function:   	void INT_T2(void)
*Description:	T2  Interrupt handler
*		 		10ms
*Input:   
*Output:     
*************************************************************************************/
void INT_T2(void) interrupt INT_VECTOR_T2
{
	if(TF2)
	{
		TF2=0;
		if(SPIOvTime!=0) SPIOvTime--;
	}
	if(EXF2)
	{
		EXF2=0;
	}
}


/*************************************************
*Function:     	void DelayXus(u8 xUs)
*Description:   	dealy,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    dealy,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:     
*************************************************************************************/
void SendByte(u8 tByte)
{
	bit bES0;
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input: 			u8* PStr:the string to be send
*Output:     
*************************************************************************************/
void SendStr(u8* PStr)
{
	while(*PStr != 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:   	void SendHex(u8 ToSend)
*Description:	send data for HEX
*Input: 			u8 ToSend: the data to be send
*Output:     
*************************************************************************************/
void SendHex(u8 ToSend)
{
	u8 x;
//	SendByte(' ');
//	SendByte('0');
//	SendByte('x');
	x = ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/***********************************************************************************
*Function:   	void SendDW2DEC(u32 ToSend)
*Description:	send data for DEC
*Input: 			u32 ToSend: the data(4 bytes) to be send
*Output:     
*************************************************************************************/
void SendDW2DEC(u32 ToSend)
{
	bit bFlag;
	u8 x;
	bFlag=0;
	x=ToSend/1000000000L;
	ToSend=ToSend%1000000000L;
	if(x!=0)
	{	bFlag=TRUE;
		SendByte('0'+x);
		SendByte(',');
	}
	
	x=ToSend/100000000L;
	ToSend=ToSend%100000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend/10000000L;
	ToSend=ToSend%10000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend/1000000L;
	ToSend=ToSend%1000000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
		SendByte(',');
	}
	
	x=ToSend/100000L;
	ToSend=ToSend%100000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend/10000L;
	ToSend=ToSend%10000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend/1000L;
	ToSend=ToSend%1000L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
		SendByte(',');
	}
	
	x=ToSend/100L;
	ToSend=ToSend%100L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend/10L;
	if((x!=0)||(bFlag==TRUE))
	{	bFlag=TRUE;
		SendByte('0'+x);
	}
	
	x=ToSend%10L;
	SendByte('0'+x);
		
}

/***********************************************************************************
*Function:   	void ResetCounter()
*Description:	reset counter
*Input:  
*Output:     
*************************************************************************************/
void ResetCounter()
{
	// Close T0,T1
	TM_GlobalSTOP(TIMER0_GLOBAL_BIT|TIMER1_GLOBAL_BIT);
	TM_SetT0HighByte(0);
	TM_SetT0LowByte(0);
	TM_SetT1HighByte(0);
	TM_SetT1LowByte(0);
	// Enable TIMER0,T1
	TM_GlobalStart(TIMER0_GLOBAL_BIT|TIMER1_GLOBAL_BIT);
}

/***********************************************************************************
*Function:   	void SendCounter()
*Description:	output counter value by UART0
*Input:  
*Output:     
*************************************************************************************/
void SendCounter()
{
	DWordTypeDef dwTime;
	TM_GlobalSTOP(TIMER0_GLOBAL_BIT|TIMER1_GLOBAL_BIT);
	dwTime.B.BHigh=TH0;
	dwTime.B.BMHigh=TL0;
	dwTime.B.BMLow=TH1;
	dwTime.B.BLow=TL1;
	SendStr("\nTIME: ");
	SendDW2DEC(dwTime.DW);
	SendStr("us");
}

/***********************************************************************************
*Function:   	void InitUart0_S0BRG(void)
*Description:   	Initialize Uart0, The source of baud rate was S0BRG
*Input:   
*Output:     
*************************************************************************************/
void InitUart0_S0BRG(void)
{	
	UART0_SetAccess_S0CR1();			// Enable access S0CR1 
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R. source:S0BRG
	UART0_SetS0BRGBaudRateX2();			// B.R. x2
	UART0_SetS0BRGSelSYSCLKdiv12();		// S0BRG clock source: SYSCLK/12

    // Sets B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_48000000_12T);
	
	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:   	void InitTimer0(void)
*Description:	Initialize Timer0
*Input:   
*Output:     
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit timer
	TM_SetT0Clock_T1OF();					// TIMER0 Clock source: T1OF
	TM_SetT0Gate_Disable();
	TM_SetT0LowByte(0);						// Sets TL0 value
	TM_SetT0HighByte(0);					// Sets TH0 value
	//TM_EnableT0();						// Enable TIMER0
}

/***********************************************************************************
*Function:   	void InitTimer1(void)
*Description:	Initialize Timer1
*Input:   
*Output:     
*************************************************************************************/
void InitTimer1(void)
{
	TM_SetT1Mode_1_16BIT_TIMER();			// TIMER1 Mode: 16-bit timer
	TM_SetT1Clock_SYSCLKDiv48();			// TIMER1 Clock source: SYSCLK/48
	TM_SetT1Gate_Disable();
	TM_SetT1LowByte(0);						// Sets TL1 value
	TM_SetT1HighByte(0);					// Sets TH1 value
	//TM_EnableT1();						// Enable T1
}

/***********************************************************************************
*Function:   	void InitTimer2(void)
*Description:	Initialize Timer2
*Input:   
*Output:     
*************************************************************************************/
void InitTimer2(void)
{
	TM_SetT2Mode_AutoRLandExI();			// TIMER2 Mode: 16-bit Auto-Reload timer
	TM_SetT2Clock_SYSCLKDiv12();			// TIMER2 Clock source: SYSCLK/12
	
	TM_SetT2LowByte(TIMER_12T_10ms_TL);		// Sets TL2 value
	TM_SetT2HighByte(TIMER_12T_10ms_TH);	// Sets TH1 value
	TM_SetT2RLHighByte(TIMER_12T_10ms_TH);	// Sets RCAP2H
	TM_SetT2RLLowByte(TIMER_12T_10ms_TL);	// Sets RCAP2L
	TM_EnableT2();							// Enable T2

	
}

/***********************************************************************************
*Function:   	void InitInterrupt(void)
*Description:	Initialize Interrupt
*Input:   
*Output:     
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER2();
	
}

/***********************************************************************************
*Function:   	void InitPort(void)
*Description:  	Initialize IO Port
*Input:   
*Output:     
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);		// Set P22,P24,P26 as Push-Pull,For LED.
	
	PORT_SetP3PushPull(BIT3);				// Set P33 as Push-Pull, For nSS
	PORT_SetP1PushPull(BIT4|BIT5|BIT7);		// Set P14(nSS),P15(MOSI),P17(SPICLK) as push-pull for output.
	PORT_SetP1OpenDrainPu(BIT6);			// Set P16(MISO) as open-drain with pull-high for digital input
}


/***********************************************************************************
*Function:   	void InitClock(void)
*Description:	Initialize clock
*Input:   
*Output:     
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 Output MCK/4
	//CLK_P60OC_MCKDiv4();
}


/***********************************************************************************
*Function:   	void InitSPI_Master(void)
*Description:	Initialize SPI Master
*Input:   
*Output:     
*************************************************************************************/
void InitSPI_Master(void)
{
	SPI_Enable();									// Enable SPI
	SPI_SelectMASTERByMSTRbit();					// Set to MASTER
	SPI_SetClock(SPI_CLK_SYSCLK_4);					// Set Clock SYSCLK/4 48M/4=12M
	SPI_SetCPOL_0();								// CPOL=0 
	SPI_SetDataMSB();								// Data MSB
	SPI_SetCPHA_0();								// CPHA=0
	SPI_SetUseP14P15P16P17();						// IO Port: nSS/P14,MOSI/P15,MISO/P16,SPICLK/P17
}

/***********************************************************************************
*Function:   	u8 SPITransceiver(u8 SPI_DATA)
*Description:	SPI Master transmit 
*Input:   		u8 SPI_DATA: Data to be send
*Output:   		u8:  Received data
*************************************************************************************/
u8 SPITransceiver(u8 SPI_DATA)
{
	SPI_SendData(SPI_DATA);							// Send data
	while(SPI_ChkCompleteFlag()==0);				// Wait complete
	SPI_ClearCompleteFlag();						// Clear flag
	return SPI_GetData();							// Return data
}

/*
*************************************************************************************
*  SPI FLASH
*
*************************************************************************************
*/


/***********************************************************************************
*Function:		void SPI_Flash_MultiBytesRead (u32 Address, u8 *BufferAddreass, u32 Length)
*Description:	Read n bytes from flash
*
*Input:	  		u32 Address: Address 0x00000000 ~ 0xFFFFFFFF
*		  		u8 *BufferAddreass: Buffer address
*		  		u16 Length: Buffer length
*Output: 
*************************************************************************************/
void SPI_Flash_MultiBytesRead (u32 Address, u8 *BufferAddreass, u16 Length)
{
	SPI_nSS = 0;
    /* Write command */
    SPITransceiver(0x03);
	
    /* Write address */
    SPITransceiver((u8)(Address>>16));
    SPITransceiver((u8)(Address>>8));
    SPITransceiver((u8)(Address));
    
    /* Read data */
    while(Length !=0)
	{
		*BufferAddreass=SPITransceiver(Dummy_Data);
        BufferAddreass ++;
        Length--;
	}
	SPI_nSS = 1;
}

/***********************************************************************************
*Function:		u32 SPI_Flash_Read_ID (void)
*Description:	read Flash ID 

*Input:	
*Output: 		uint32_t: Flash ID 
*************************************************************************************/
u32 SPI_Flash_Read_ID (void)
{
	DWordTypeDef RDAT;
    
    SPI_nSS = 0;
    /* Write command */
    SPITransceiver(0x9F);
    
    /* Read ID */
    RDAT.B.BMHigh=SPITransceiver(Dummy_Data);
    RDAT.B.BMLow=SPITransceiver(Dummy_Data);
    RDAT.B.BLow=SPITransceiver(Dummy_Data);
	SPI_nSS  = 1;
    
    return RDAT.DW;
}

/***********************************************************************************
*Function:		u8 SPI_Flash_Write_Enable (void)
*Description: 	Enable write Flash

*Input:	
*Output: 		0: success  other:fail
*************************************************************************************/
u8 SPI_Flash_Write_Enable(void)
{
	u8 RDAT;

	// set over time 2*10=20ms
    SPIOvTime=2;
  Re_Flash_Write_Enable:
    /* Write enable */
	SPI_nSS  = 0;
	SPITransceiver(0x06);
	SPI_nSS  = 1;

	nop();
	nop();
	nop();
	
    /* Read status register */
	SPI_nSS  = 0;
	SPITransceiver(0x05);							//RDSR
   
	RDAT = SPITransceiver(Dummy_Data);              // Get received data

	SPI_nSS  = 1;
	
    /* Check WEL == 1 */
	if((RDAT & 0x02) == 0x00)
	{
		if(SPIOvTime!=0)
		{
			goto Re_Flash_Write_Enable;
		}
		else
		{ // over time
			return 1;
		}
	}
	else
	{
		return 0;
	}
}

/***********************************************************************************
*Function:		u8 SPI_Flash_Check_Busy(void)
*Description:	check busy 

*Input:	
*Output: 		0: Free  1:busy
*************************************************************************************/
u8 SPI_Flash_Check_Busy(void)
{
	u8 RDAT;
    
    /* Read status register */
	SPI_nSS  = 0;
	SPITransceiver(0x05);							//RDSR

	RDAT = SPITransceiver(Dummy_Data);              // Get received data
	
    /* Check erase or write complete */
	SPI_nSS  = 1;
	if((RDAT & 0x01) == 0x01)						//WIP, write in progress
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/***********************************************************************************
*Function:		void SPI_Flash_Write_Status(u8 uStatus)
*Description:	Write status

*Input:			u8 uStatus:status

*Output: 
*************************************************************************************/ 
void SPI_Flash_Write_Status(u8 uStatus)
{
    SPI_Flash_Write_Enable();
    
	SPI_nSS  = 0;
	SPITransceiver(0x01);                           // Write Status Register(WRSR) command 
                
	
    /* Write Status */
    SPITransceiver(uStatus);
	
	SPI_nSS  = 1;
}

/***********************************************************************************
*Function:		u8 SPI_Flash_Page_Program(u32 Addr, u8 *DataSource, u16 Length)
*Description:	Write n bytes to Flash

*Input:			u32 Addr: Start address. 0x00000000 ~ 0xFFFFFFFF.
*				u8 *DataSource: Buffer address
*				u16 Length: Buffer length, MAX.256 bytes
*Output: 		0: success  other:fail
*************************************************************************************/ 
u8 SPI_Flash_Page_Program(u32 Addr, u8 *DataSource, u16 Length)
{
    if(SPI_Flash_Write_Enable()==1)
    {
		return 1;
    }
    
	SPI_nSS  = 0;
	SPITransceiver( 0x02);                          // Write command 
	
    /* Write address */
    SPITransceiver((u8)(Addr>>16));
    SPITransceiver((u8)(Addr>>8));
    SPITransceiver((u8)(Addr));
	
	while(Length!=0)                                // Write Data
	{
		SPITransceiver(*DataSource);
        DataSource++;
		Length--;
	}
	
	SPI_nSS  = 1;
	
	// set over time 2*10=20ms
    SPIOvTime=2;
  	while(SPIOvTime!=0)
  	{
		if(SPI_Flash_Check_Busy()==0)                // Wait program complete
		{
			return 0;
		}
	}
	return 2;
}

/***********************************************************************************
*Function:		u8 SPI_Flash_Block_Erase(u32 Addr)
*Description:	Flash block erase

*Input:			u32 Addr: start addresss.
*Output: 		0: success  other:fail
*************************************************************************************/ 
u8 SPI_Flash_Block_Erase(u32 Addr)
{
    if(SPI_Flash_Write_Enable()==1)
    {
		return 1;
    }
    
	SPI_nSS  = 0;
	SPITransceiver(0xD8);                           // Block Erase command 
                
	
    /* Write address */
    SPITransceiver((u8)(Addr>>16));
    SPITransceiver((u8)(Addr>>8));
    SPITransceiver((u8)(Addr));
	
	SPI_nSS  = 1;
   
	// set over time 210*10=2.1s
    SPIOvTime=210;
  	while(SPIOvTime!=0)
  	{
		if(SPI_Flash_Check_Busy()==0)               // Wait Block Erase complete
		{
			return 0;
		}
	}
	return 2;
}


/*
*************************************************************************************
*/ 


/***********************************************************************************
*Function:   void InitSystem(void)
*Description:   Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitSPI_Master();
	InitUart0_S0BRG();
	InitTimer0();
	InitTimer1();
	InitTimer2();
	InitInterrupt();
}


void main()
{
	u32 i;
	bit bOK;
	DWordTypeDef dwTemp;
	
    InitSystem();

    INT_EnAll();
	
	SendStr("\nHello!");
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=0;LED_G_1=1;LED_R=1;

	
	
	dwTemp.DW=SPI_Flash_Read_ID();
	SendStr("\nFlash ID: ");
	SendHex(dwTemp.B.BMHigh);
	SendHex(dwTemp.B.BMLow);
	SendHex(dwTemp.B.BLow);

	// Clear write protect
	SPI_Flash_Write_Status(0x00);

	if(dwTemp.DW == FLASH_ID)
	{
		bOK=TRUE;
		// Test
		SendStr("\nFlash Erase...");
		i=FLASH_TEST_SIZE_MAX/FLASH_BLOCK_SIZE;
		// Erase Flash
		dwTemp.DW=FLASH_TEST_START_ADDRESS;
		ResetCounter();
		while(i!=0)
		{
			LED_G_1 = !LED_G_1;
			if(SPI_Flash_Block_Erase(dwTemp.DW)!=0)
			{
				bOK=FALSE;
				break;
			}
			dwTemp.DW=dwTemp.DW+FLASH_BLOCK_SIZE;
			i--;
		}
		SendCounter();
		for(i=0;i<256;i++)
		{
			TestBuf[i]=i;
		}
		// Write Flash
		SendStr("\nFlash Write 1M");
		i=FLASH_TEST_SIZE_MAX/256;		
		dwTemp.DW=FLASH_TEST_START_ADDRESS;
		ResetCounter();
		while(i!=0)
		{
			LED_G_1 = !LED_G_1;
			if(SPI_Flash_Page_Program(dwTemp.DW,&TestBuf[0],256)!=0)
			{
				bOK=FALSE;
				break;
			}
			dwTemp.DW=dwTemp.DW+256;
			i--;
		}
		SendCounter();
		// Read Flash
		SendStr("\nFlash Read 1M");
		i=FLASH_TEST_SIZE_MAX/512;		
		dwTemp.DW=FLASH_TEST_START_ADDRESS;
		ResetCounter();
		while(i!=0)
		{
			LED_G_1 = !LED_G_1;
			SPI_Flash_MultiBytesRead(dwTemp.DW,&TestBuf[0],512);
			dwTemp.DW=dwTemp.DW+512;
			i--;
		}
		SendCounter();
	}
	else
	{
		SendStr("\nFlash isn't MX25L3206! ");
		bOK=FALSE;
	}
	while(1)
	{
		DelayXms(100);
		if(bOK)
		{
			LED_G_0 = !LED_G_0;
		}
		else
		{
			LED_R = !LED_R;
		}
	}
	
}


