/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=24MHz, SysCLK=24MHz
*    Description:
*			KBI Wakeup
*
*    Note:
*
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

/***********************************************************************************
*Function:   	void INT_KBI(void)	
*Description:	KBI Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_KBI(void)		interrupt INT_VECTOR_KB
{
	if(KBCON&KBIF==KBIF)	//KBIF is set,KBI interrupt occured
	{
		LED_R=0;
		KBCON&=(~KBIF);		//clear KBI interrupt flag
		LED_R=1;
	}
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:		u8 IAP_ReadPPage(u8 PsfrAddr)
*Description:	Read P page sfr
*Input:   		u8 PsfrAddr: sfr Address
*Output:        u8 : sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8 PsfrAddr)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*		 		u8 PsfrData: sfr data
*Output:     
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnKBI();				// Enable KBI interrupt
}	

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2OpenDrain(BIT4|BIT5|BIT6|BIT7);			// Set P24,P25,P26,P27 as open drain
	PORT_EnP24P25P26P27PullHigh();						// Enable P24,P25,P26,P27 pull-high

	PORT_SetP1OpenDrain(BIT0|BIT1);						// Set P10 as open drain
	PORT_EnP10P11P12P13PullHigh();						// Enable P10,P11,P12,P13 pull-high
}

/***********************************************************************************
*Function:   	void InitKBI(void)
*Description:   	Initialize KBI
*Input:   
*Output:     
*************************************************************************************/
void InitKBI(void)
{
	KBI_SetKBMask(KBI4|KBI5);				//Enable channel KBI4,KBI5
	KBI_SetKBI4567_P10P11P12P13();			//KBI4,KBI5 set to P10,P11
	KBI_SetKBPattern(KBI4_1|KBI5_1);		//KBI Pattern is 1
	KBI_SetPatternNotEqual();				//Pattern Matching polarity selection : not equal
}

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitKBI();
	InitInterrupt();

	INT_EnAll();  // Enable global interrupt
}


void main()
{

	u8 KBICnt=0;

	u8 BakCKCON2;
	
    InitSystem();

	
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=1;LED_G_1=1;LED_R=1;

    while(1)
    {
		KBICnt=20;
		while(KBICnt!=0)
		{
			LED_G_0=!LED_G_0;
			DelayXms(100);
			KBICnt--;
		}
		LED_G_0=1;
		
		// MCK=OSCin
		BakCKCON2=IAP_ReadPPage(CKCON2_P);							// Backup CKCON2 (if used PLL CKM)
		IAP_WritePPage(CKCON2_P,BakCKCON2&(~(MCKS0_P|MCKS1_P)));	// MCK=OSCin (if used PLL CKM)

		POW_SetMode_PD();											// MCU enter Power-Down mode
		
		_nop_();
		DelayXus(100);												// delay for stabilize CKM (if used PLL CKM)
		IAP_WritePPage(CKCON2_P,BakCKCON2);							// restore CKCON2,use CKM (if used PLL CKM)

		LED_G_1=0;
		DelayXms(500);
		LED_G_1=1;
		DelayXms(500);
		LED_G_1=0;
		DelayXms(500);
		LED_G_1=1;
    }
}





