/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			CPUCLK=SYSCLK (12M)
*			PCA Clock: SYSCLK
*			CH0(P22),CH1(P33),CH2(P24),CH3(P34),CH4(P26),CH5(P35),PWM6(P20),PWM7(P21)
*			Change duty in PCA interrupt
*			Edge-Aligned,Resolution is 1000
*			PWM frequency: PCAClk/(1000)=12MHz/1000=12KHz
*
*			P10 as input to control PWM output.P10=1: Enable P10=0: Disable
*	
*    Note:
*			Change duty in PCA interrupt while using unbuffer mode
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

#define PWM_MIN			(0*1)
#define PWM_MAX			(1000*1)
#define PWM_3_4			(750*1)
#define PWM_2_4			(500*1)
#define PWM_1_4			(250*1)
#define PWM_LOW			(40*1)
#define PWM_HINT		(50*1)

#define PCA_RELOAD		(PWM_MAX)

#define PCA_C           (65536)       	

#define PCA_CL(x)		(u8)((~(u16)x)+1) 
#define PCA_CH(x)     	(u8)(((~(u16)x)+1)>>8)      

xdata WordTypeDef wDuty[8];
bit bDutyChange;
u8 DutyFlag;

/***********************************************************************************
*Function:    void INT_PCA(void)
*Description: PCA Interrupt handler
*		 
*Input:   
*Output:     
*************************************************************************************/
void INT_PCA(void) interrupt INT_VECTOR_PCA
{
	WordTypeDef duty;
	if(CF)
	{
		CF=0;
		// Todo...
		// ......
		if(bDutyChange)
		{
			duty.W=PCA_C-wDuty[0].W;
			PCA_CH0_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[1].W;
			PCA_CH1_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[2].W;
			PCA_CH2_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[3].W;
			PCA_CH3_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[4].W;
			PCA_CH4_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[5].W;
			PCA_CH5_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[6].W;
			PCA_CH6_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PCA_C-wDuty[7].W;
			PCA_CH7_SetValue(duty.B.BHigh,duty.B.BLow);
			bDutyChange=FALSE;
		}
	}
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	P2=B10101000;									// PWM output low level
	P3=B11000111;
	PORT_SetP2PushPull(BIT0|BIT1|BIT2|BIT4|BIT6);	// set P20(PWM6),P21(PWM7),P22(CEX0),P24(CEX2),P26(CEX4) as push-pull for PWM output
	PORT_SetP3PushPull(BIT3|BIT4|BIT5);				// set P33(CEX1),P34(CEX3),P35(CEX5) as push-pull for PWM output

	P10=1;
	PORT_SetP1OpenDrainPu(BIT0);					// Set P10 as open-drain with pull-high
}

/***********************************************************************************
*Function:   	void InitPCA_PWM(void)
*Description:	Initialize PCA for PWM
*Input:   
*Output:   		
*************************************************************************************/
void InitPCA_PWM(void)
{
	PCA_SetCLOCK_SYSCLK();				// PCA clock: SysClk
	
	PCA_CH0_SetMode_PWM();
	PCA_CH1_SetMode_PWM();
	PCA_CH2_SetMode_PWM();
	PCA_CH3_SetMode_PWM();
	PCA_CH4_SetMode_PWM();
	PCA_CH5_SetMode_PWM();
	PCA_CH6_SetMode_PWM();
	PCA_CH7_SetMode_PWM();

	PCA_CH0_SetPWM_16Bit();
	PCA_CH1_SetPWM_16Bit();
	PCA_CH2_SetPWM_16Bit();
	PCA_CH3_SetPWM_16Bit();
	PCA_CH4_SetPWM_16Bit();
	PCA_CH5_SetPWM_16Bit();
	PCA_CH6_SetPWM_16Bit();
	PCA_CH7_SetPWM_16Bit();

	PCA_SetPWM_EdgeAligned();			// Edge-aligned

	// Set PWM duty
	PCA_CH0_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH1_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH2_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH3_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH4_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH5_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH6_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	PCA_CH7_SetValue(PCA_CH(PWM_MIN),PCA_CL(PWM_MIN));
	
	// Set Freq
	PCA_SetCounter(PCA_C-PCA_RELOAD);
	PCA_SetCounterReload(PCA_C-PCA_RELOAD);

	// Disable PWM output
	PCA_SetPWM0_DisOutput();					
	PCA_SetPWM1_DisOutput();
	PCA_SetPWM2_DisOutput();					
	PCA_SetPWM3_DisOutput();
	PCA_SetPWM4_DisOutput();					
	PCA_SetPWM5_DisOutput();
	PCA_SetPWM6_DisOutput();					
	PCA_SetPWM7_DisOutput();

	PCA_SetCEX1CEX3CEX5_P33P34P35();	// Set CEX1:P33,CEX3:P34,CEX5:P35
	
	PCA_CF_EnInterrupt();				// Enable PCA CF interrupt
	
	PCA_EnPCACounter();					// Enable PCA counter 
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnPCA();					// Enable PCA interrupt
	
}	

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
#if (MCU_SYSCLK==11059200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==12000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);
	
#else
	// SysClk=12MHz CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if (MCU_SYSCLK==22118400)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==24000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==29491200)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==32000000)
#if (MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if (MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if (MCU_SYSCLK==48000000)
	// SysClk=48MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitClock();
	InitPort();
	InitPCA_PWM();
	InitInterrupt();
}


void main()
{
	u8 i,x,cnt;
	bit bOutFlag;
    InitSystem();

	LED_G_0=0;LED_R=0;LED_G_0=0;
	DelayXms(1000);
	LED_G_0=1;LED_R=1;LED_G_1=1;

	INT_EnAll();
	

	wDuty[0].W=PWM_MIN;
	wDuty[1].W=PWM_LOW;
	wDuty[2].W=PWM_1_4;
	wDuty[3].W=PWM_2_4;
	wDuty[4].W=PWM_3_4;
	wDuty[5].W=PWM_2_4;
	wDuty[6].W=PWM_1_4;
	wDuty[7].W=PWM_MAX;
	
	DutyFlag=0x00;
	bOutFlag=FALSE;
	
	while(1)
    {
    	DelayXms(1);
    	cnt++;
    	if(cnt>=200)
    	{
			cnt=0;
	    	LED_G_1=!LED_G_1;
	    	if(bOutFlag==TRUE)
	    	{
				while(bDutyChange==TRUE);		//wait for the previous data is complete
		    	x=0x01;
		    	for(i=0;i<8;i++)
		    	{
					if((DutyFlag&x)==0)
					{
						wDuty[i].W=wDuty[i].W+20;
						if(wDuty[i].W >= PWM_MAX)
						{
							wDuty[i].W = PWM_MAX;
							DutyFlag=DutyFlag|x;
						}
					}
					else
					{
						if(wDuty[i].W < 21)
						{
							wDuty[i].W = PWM_MIN;
							DutyFlag=DutyFlag&(~x);
						}
						else
						{
							wDuty[i].W=wDuty[i].W-20;
						}
					}
					x=x<<1;
		    	}
		    	bDutyChange=TRUE;
	    	}
    	}
    	if(P10==1)
    	{
			if(bOutFlag==FALSE)
			{
				bOutFlag=TRUE;
				LED_G_0=0;
				// Enable PWM output
				PCA_SetPWM0_EnOutput(); 				
				PCA_SetPWM1_EnOutput();
				PCA_SetPWM2_EnOutput(); 				
				PCA_SetPWM3_EnOutput();
				PCA_SetPWM4_EnOutput(); 				
				PCA_SetPWM5_EnOutput();
				PCA_SetPWM6_EnOutput(); 				
				PCA_SetPWM7_EnOutput();
			}
    	}
    	else
    	{
			if(bOutFlag==TRUE)
			{
				bOutFlag=FALSE;
				LED_G_0=1;
				// Disable PWM output
				PCA_SetPWM0_DisOutput();					
				PCA_SetPWM1_DisOutput();
				PCA_SetPWM2_DisOutput();					
				PCA_SetPWM3_DisOutput();
				PCA_SetPWM4_DisOutput();					
				PCA_SetPWM5_DisOutput();
				PCA_SetPWM6_DisOutput();					
				PCA_SetPWM7_DisOutput();
			    P2=B10101000;									// PWM output low level
			    P3=B11000111;
			}
    	}
    }
}








