#include "stdafx.h"
#include "GenericFunc.h"
#include "HexFile.h"

extern DOWNLOAD_FILE g_DownloadFile;

//-----------------------------------------------------------------------------
// Function Name : LoadFile
// Input         : lpFilePath, The path to the file
//                 
// Return        : dwResult
//-----------------------------------------------------------------------------
// Description   : 1. Enumerate all of the HID devices 
//                 2. Open every HID device and get HID attribute
//                 3. Compare with the specified Vendor ID
//                 4. List the Product ID in pComboBox
//-----------------------------------------------------------------------------
DWORD LoadFile(LPTSTR lpFilePath)
{    
    DWORD dwResult = ERROR_INVALID_DATA;
    PTCHAR ptch = NULL;

    OutputDebugString(lpFilePath);

    g_DownloadFile.m_bValid = FALSE;
    g_DownloadFile.m_dwFileSize = 0;
    g_DownloadFile.m_dwChecksum = 0;
    g_DownloadFile.m_szFilePathName[0] = 0;

    if (g_DownloadFile.m_pBuffer != NULL)
        delete []g_DownloadFile.m_pBuffer;
    
    ptch = lpFilePath + _tcslen(lpFilePath) - 3;
    
    if (_tcsicmp(ptch, _T("hex")) == 0)
    {
        CHexFile hFile;
        if (hFile.Open(lpFilePath))
        { 
            g_DownloadFile.m_bValid = TRUE;
            g_DownloadFile.m_dwFileSize = hFile.GetLength();
            _tcscpy(g_DownloadFile.m_szFilePathName, lpFilePath);
            g_DownloadFile.m_pBuffer = new BYTE[g_DownloadFile.m_dwFileSize];            
            hFile.Read(g_DownloadFile.m_pBuffer, g_DownloadFile.m_dwFileSize);
            hFile.Close();            
        }
    }
    else
    {
        CFile hFile;
        if (hFile.Open(lpFilePath, CFile::modeRead))
        {
            g_DownloadFile.m_bValid = TRUE;
            g_DownloadFile.m_dwFileSize = hFile.GetLength();
            _tcscpy(g_DownloadFile.m_szFilePathName, lpFilePath);
            g_DownloadFile.m_pBuffer = new BYTE[g_DownloadFile.m_dwFileSize];            
            hFile.Read(g_DownloadFile.m_pBuffer, g_DownloadFile.m_dwFileSize);
            hFile.Close();
        }
    }
  
    if (g_DownloadFile.m_bValid == TRUE)
    {
		__asm
		{
			mov ecx, g_DownloadFile.m_dwFileSize
			mov edx, g_DownloadFile.m_pBuffer
			add edx, ecx
L_Next:		dec	edx
			movzx eax, byte ptr[edx]
			add	g_DownloadFile.m_dwChecksum, eax
			loop L_Next
		}
        dwResult = ERROR_SUCCESS;
    }
    else
    {
        dwResult = ERROR_OPEN_FAILED;          
    } 
    return dwResult;
}

//-----------------------------------------------------------------------------
// Function Name : GetFileLastWriteTime
// Input         : pDownloadFile, pointer to a struct of DOWNLOAD_FILE
//                 
// Return        : dwResult
//-----------------------------------------------------------------------------
// Description   : 1. Get the last write time 
//                 2. Convert a file time to a local file time
//                 3. Convert a file time to system time
//-----------------------------------------------------------------------------
DWORD GetFileLastWriteTime(DOWNLOAD_FILE *pDownloadFile)
{
    FILETIME ftLocal;

	HANDLE hFile = INVALID_HANDLE_VALUE;

    hFile = CreateFile( pDownloadFile->m_szFilePathName,
		 			    GENERIC_READ,					
					    FILE_SHARE_READ | FILE_SHARE_WRITE,
						NULL,
						OPEN_EXISTING,
						FILE_FLAG_OVERLAPPED,  
						NULL ); 

    pDownloadFile->m_PreLastWriteTime = pDownloadFile->m_CurLastWriteTime;

    if (!GetFileTime(hFile, NULL, NULL, &pDownloadFile->m_CurLastWriteTime))
		return FALSE;

    FileTimeToLocalFileTime(&pDownloadFile->m_CurLastWriteTime, &ftLocal);
    FileTimeToSystemTime(&ftLocal, &pDownloadFile->m_LastWriteTime);

	CloseHandle(hFile);
	hFile = NULL;

    return ERROR_SUCCESS;
}

//-----------------------------------------------------------------------------
// Function Name : ExtractResourceToFile
// Input         : hModule    
//               : lpszType
//               : nResID
//               : lpszFileName            
// Return        : DWORD
//-----------------------------------------------------------------------------
// Description   : Extract the resource with the specified type in current application and 
//                 save as a specified file name
//-----------------------------------------------------------------------------
DWORD ExtractResourceToFile(HMODULE hModule, LPCTSTR lpszType, UINT nResID, LPCTSTR lpszFileName)
{
    HRSRC hResInfo = NULL;
    HGLOBAL hGlobal = NULL;
    LPVOID lpResLock = NULL;
    DWORD dwSizeOfRes = 0;
    HANDLE hFile = INVALID_HANDLE_VALUE;
    BOOL bResult = FALSE;
    DWORD dwResult = ERROR_SUCCESS;
    DWORD dwWritten = 0;;

    hResInfo = ::FindResource(hModule, MAKEINTRESOURCE(nResID), lpszType);
    if (hResInfo != NULL)
    {
        dwSizeOfRes = ::SizeofResource(hModule, hResInfo);
        if (dwSizeOfRes != 0)
        {
            hGlobal = ::LoadResource(hModule, hResInfo);
            if (hGlobal != NULL)
            {
                lpResLock = ::LockResource(hGlobal);
                if (lpResLock != NULL)
                {
                    hFile = ::CreateFile(lpszFileName,   
                                         GENERIC_WRITE,
                                         FILE_SHARE_WRITE | FILE_SHARE_READ,
                                         NULL,
                                         CREATE_ALWAYS,
                                         0,
                                         NULL);
                    if (hFile != INVALID_HANDLE_VALUE)
                    {
                        bResult = ::WriteFile(hFile, lpResLock, dwSizeOfRes, &dwWritten, NULL);
                        if ((bResult != 0) && (dwWritten == dwSizeOfRes))
                        {
                            bResult = ::FlushFileBuffers(hFile);
                            if (bResult != 0)
                            {
                               bResult = ::CloseHandle(hFile);
                               if (bResult != 0)
                                   bResult = TRUE;
                            }
                        }
                    }
                }
            }
        }
    }
    if (bResult != TRUE)
        dwResult = GetLastError();  
    return dwResult;
}

//-----------------------------------------------------------------------------
// Function Name : GetAppFilePath
// Input         : lpFilePath, the pointer to a buffer
//                 dwSize, the size of the buffer
// Return        : dwResult
//-----------------------------------------------------------------------------
// Description   : Retrieve the fully path of current application
//                                  
//-----------------------------------------------------------------------------
DWORD GetAppFilePath(LPTSTR lpFilePath, DWORD dwSize)
{
    DWORD dwResult = ERROR_SUCCESS;
    TCHAR *pch = NULL;

    dwResult = ::GetModuleFileName(NULL, lpFilePath, dwSize);
    if (dwResult == 0)
        dwResult = ::GetLastError();
    else
    {
        pch = ::_tcsrchr(lpFilePath, '\\');
        if (pch != NULL)
            *(pch+1) = '\0';

        dwResult = ERROR_SUCCESS;
    }

    return dwResult;
}

//-----------------------------------------------------------------------------
// Function Name : IsFileExist
// Input         : lpFileName, the pointer to a file with full path name
//
// Return        : bResult
//-----------------------------------------------------------------------------
// Description   : use windows API "CreaeFile" to check if file is exist
//                                  
//-----------------------------------------------------------------------------
BOOL IsFileExist(LPCTSTR lpFileName)
{
    HANDLE hFile = INVALID_HANDLE_VALUE;
    BOOL bResult = FALSE;    

    hFile = CreateFile(lpFileName,
                       0,
                       FILE_SHARE_READ | FILE_SHARE_WRITE,
                       NULL,
                       OPEN_EXISTING,
                       0,
                       NULL);

    if (hFile != INVALID_HANDLE_VALUE)
    {
        CloseHandle(hFile);
        bResult = TRUE;
    }
    
    return bResult;
}

//-----------------------------------------------------------------------------
// Function Name : SelectFilePath
// Input         : lpFilePath, the pointer to a buffer
//                 bOpen, a flag to determine if it will be opened or saved
// Return        : bResult
//-----------------------------------------------------------------------------
// Description   : If bOpen is TRUE, display a File Open dialog to allow users 
//                 to select a specified file.
//                 If bOpen is FALSE, display a File Save As dialog to allow 
//                 users to select a file to save.                 
//-----------------------------------------------------------------------------
DWORD SelectFilePath(LPTSTR lpFilePath, BOOL bOpen)
{
    TCHAR szFilters[] = _T("Binary files (*.bin)|*.bin|All files (*.*)|*.*||");
    CFileException ex;
    CString strTmp = _T("");
    CFile TempFile;
    DWORD dwLength = 0;

    CFileDialog dlgOpen(bOpen, _T("All files"), _T("*.*"), OFN_FILEMUSTEXIST | OFN_HIDEREADONLY, szFilters);

    if (dlgOpen.DoModal() == IDOK)
    {
        strTmp = _T("");
        strTmp = dlgOpen.GetPathName();       
        if (bOpen == TRUE)
        {
            if ( !TempFile.Open(strTmp, CFile::modeRead | CFile::shareDenyWrite, &ex)) 
            {            
                 TCHAR szError[1024];
                 ex.GetErrorMessage(szError, 1024); 
                 ShowErrorMessage(NULL, _T("ERROR"), 0, szError);                
                 return -1;
            }   
            else
            {           
                TempFile.Close();
                _tcscpy(lpFilePath, strTmp.GetBuffer(strTmp.GetLength()));
                dwLength = _tcslen(lpFilePath);
                return dwLength;
            }
        }
        else
        {   
            if (!TempFile.Open( strTmp, CFile::modeRead | CFile::shareDenyWrite, &ex)) 
            {
                if (ex.m_cause == CFileException::fileNotFound)
                {               
                    if (!TempFile.Open(strTmp, CFile::modeCreate | CFile::modeWrite | CFile::shareDenyWrite, &ex) ) 
                    {            
                         TCHAR szError[1024];
                         ex.GetErrorMessage(szError, 1024);                         
                         ShowErrorMessage(NULL, _T("ERROR"), 0, szError);                
                         return -1;
                    }   
                    else
                    {               
                        TempFile.Close();
                        _tcscpy(lpFilePath, strTmp.GetBuffer(strTmp.GetLength()));
                        dwLength = _tcslen(lpFilePath);
                        return dwLength;
                    }
                }
                else
                {               
                    TempFile.Close();
                    _tcscpy(lpFilePath, strTmp.GetBuffer(strTmp.GetLength()));
                    dwLength = _tcslen(lpFilePath);
                    return dwLength;
                }
            }
            else
            {               
                TempFile.Close();
                _tcscpy(lpFilePath, strTmp.GetBuffer(strTmp.GetLength()));
                dwLength = _tcslen(lpFilePath);
                return dwLength;
            }
        }
    }
    else
    {
        lpFilePath[0] = '\0';
        dwLength = _tcslen(lpFilePath);
        return dwLength;
    }       
}

//-----------------------------------------------------------------------------
// Function Name : ShowErrorMessage
// Input         : hWnd
//                 lpTilte
//                 dwErrorCode
//                 lpUserDescription
// Return        : dwResult
//-----------------------------------------------------------------------------
// Description   : Show a Message-Box with error string
//-----------------------------------------------------------------------------
DWORD ShowErrorMessage(HWND hWnd, LPCTSTR lpTitle, DWORD dwErrorCode, LPCTSTR lpUserDescription)
{
    DWORD dwResult = ERROR_SUCCESS; 
    LPVOID lpMsgBuf = NULL; 
    DWORD dwLength = 0;
    PTCHAR pDescriptionBuffer = NULL;

    if (dwErrorCode == 0)
        dwErrorCode = GetLastError();
    
    if (lpUserDescription == NULL) 
    {        
        dwResult = GetErrorMessageDescription(&lpMsgBuf, dwErrorCode);
        dwLength = _tcslen((LPCTSTR)lpMsgBuf) + 30;     
    }
    else
        dwLength = _tcslen(lpUserDescription) + 1;

    pDescriptionBuffer = new TCHAR[dwLength];

    if (pDescriptionBuffer == NULL)
    {
        if (hWnd == NULL)
            MessageBox(NULL, _T("Memory is not enough !"), lpTitle, MB_OK|MB_ICONERROR);
        else
            MessageBox(hWnd, _T("Memory is not enough !"), lpTitle, MB_OK|MB_ICONERROR);
    }
    else
    {
        if (lpUserDescription == NULL)
            _stprintf(pDescriptionBuffer, _T("Error code = 0x%04X\n\n%s"), dwErrorCode, (LPCTSTR)lpMsgBuf);
        else
            _stprintf(pDescriptionBuffer, _T("%s"), lpUserDescription);
   
        if (hWnd == NULL)
            MessageBox(NULL, pDescriptionBuffer, lpTitle, MB_OK|MB_ICONERROR);
        else
            MessageBox(hWnd, pDescriptionBuffer, lpTitle, MB_OK|MB_ICONERROR);

        delete []pDescriptionBuffer;
    }
    LocalFree(lpMsgBuf);    
    return ERROR_SUCCESS;
}

//-----------------------------------------------------------------------------
// Function Name : GetErrorMessageDescription
// Input         : lppMsgBuf, the address of the pointer passed by caller function
//                 dwErrorCode, the last-error code value                
// Return        : dwResult
//-----------------------------------------------------------------------------
// Description   : Obtain error message strings for the system error code
//-----------------------------------------------------------------------------
DWORD GetErrorMessageDescription(LPVOID *lppMsgBuf, DWORD dwErrorCode)
{    
    DWORD dwResult = ERROR_SUCCESS;
 
    dwResult = FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                             NULL,                 
                             dwErrorCode,
                             MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                             (LPTSTR)lppMsgBuf,
                             0,
                             NULL );
    
    if (dwResult == 0)
        dwResult = GetLastError();

    return dwResult;
}
