/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=24MHz, SysCLK=24MHz
*    Description:
*			RTC Clock source: XTAL 32.768KHz
*			RTC Clock prescale: 16384
*			RTC Reload Value: (64-1)=63
*			RTC Overflow freq: 32.768K/16384/1 = 2Hz
*			Enter Power-Down mode, wake up by RTC
*    Note:
*		
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

/***********************************************************************************
*Function:   	void INT_SF(void)
*Description:	SF(system flag)Interrupt handler
*		 		RTC,WDTF,BOD0F,BOD1F
*Input:   
*Output:     
*************************************************************************************/
void INT_SF(void) interrupt INT_VECTOR_SF
{
	if((PCON1&WDTF)!=0)
	{
		PCON1=WDTF;
	}
	if((PCON1&BOF0)!=0)
	{
		PCON1=BOF0;
	}
	if((PCON1&BOF1)!=0)
	{
		PCON1=BOF1;
	}
	if((PCON1&RTCF)!=0)
	{
		PCON1=RTCF;
		LED_R=!LED_R;
	}
}

/***********************************************************************************
*Function:		u8 IAP_ReadPPage(u8 PsfrAddr)
*Description:	Read P page sfr
*Input:   		u8 PsfrAddr: sfr Address
*Output:        u8 : sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8 PsfrAddr)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*		 		u8 PsfrData: sfr data
*Output:     
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitRTC(void)
*Description:	Initialize RTC 
*		RTC Clock source: XTAL 32.768KHz
*		RTC Clock prescale: 16384
*		RTC Reload Value: (64-1)=63
*		RTC Overflow freq: 32.768K/16384/1 =2Hz 0.5s
*Input:   
*Output:     
*************************************************************************************/
void InitRTC(void)
{	
	RTC_Disable();
	RTC_SetClock_ECKI_P60_Div_16384();		// RTC Clock source: P60/16384
	RTC_SetReload(64-1);					// RTC Reload Value: (64-1)=63
	RTC_SetCounter(64-1);

	RTC_ClearRTCF();			// clear RTCF
	RTC_Enable();				// Enable RTC
	
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnSF_RTC();			// Enable RTC interrupt, must enble SF interrupt
	INT_EnSF();				// Enable SF interrupt
}	

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2OpenDrain(0xFF);							// Set P2 as open drain
	PORT_EnP24P25P26P27PullHigh();						// Enable P24,P25,P26,P27 pull-high
}

/***********************************************************************************
*Function:   	void InitClock()
*Description:	Initialize clock
*Input:   
*Output:   		
*************************************************************************************/
void InitClock(void)
{
	// SysClk=24MHz CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// Enable xtal 32768,IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(GAIN_FOR_32K|ENABLE_XTAL|ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
	
	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	
	InitPort();

	InitClock();
	
	InitRTC();			

	InitInterrupt();		// Initialize Interrupt
	
	INT_EnAll();			// Enable global interrupt

}

void main()
{
	u8 i;
	
	u8 BakCKCON2;			// Backup CKCON2 (if used PLL CKM)
	
    InitSystem();
	
 	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
 	LED_G_0=1;LED_G_1=1;LED_R=1;

					
    while(1)
    {
    	for(i=0;i<50;i++)
    	{
	    	DelayXms(100);
	    	LED_G_0=!LED_G_0;
    	}
		LED_G_0=1;
    	LED_G_1=0;
		RTC_Disable();
		RTC_SetReload(64-10);					// RTC reload value,overflow time 0.5s*10=5s
		RTC_SetCounter(64-10);
		RTC_ClearRTCF();						// Clear RTCF
		RTC_Enable();							// Enable RTC
    	LED_R=1;

		// MCK=OSCin
		BakCKCON2=IAP_ReadPPage(CKCON2_P);							// Backup CKCON2 (if used PLL CKM)
		IAP_WritePPage(CKCON2_P,BakCKCON2&(~(MCKS0_P|MCKS1_P)));	// MCK=OSCin (if used PLL CKM)

    	POW_SetMode_PD();	// Enter POWER_DOWM mode
    	_nop_();

		DelayXus(100);												// delay for stabilize CKM (if used PLL CKM)
		IAP_WritePPage(CKCON2_P,BakCKCON2);							// restore CKCON2,use CKM (if used PLL CKM)

    	LED_G_1=1;
		RTC_Disable();
		RTC_SetReload(64-1);					// RTC reload value,overflow time 0.5s*10=5s
		RTC_SetCounter(64-1);
		RTC_ClearRTCF();						// Clear RTCF
		RTC_Enable();							// Enable RTC
    	
    }
}






