/*********************************************************************
*    Project: MG82F6D16-DEMO
*            MG82F6Dxx	SSOP28 EV Board (TH189A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			RTC Clock source: XTAL 32.768KHz
*			RTC Clock prescale: 32768
*			RTC Reload Value: (64-1)=63
*			RTC Overflow freq: 32.768K/32768/1 = 1Hz
*			
*			Enter Power-Down mode, wake up by RTC,KBI,exINT0
*    Note:
* 
*    Creat time:
*    Modify:
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F6D16.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6D16.H"
#include ".\include\API_Uart_BRGRL_MG82F6D16.H"


/*************************************************
*Set SysClk (MAX.50MHz)
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.25MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P22
#define LED_R		P24
#define LED_G_1		P26

u8 WakeFlag;
enum{
	WAKE_BY_RTC=0,
	WAKE_BY_ExINT0,
	WAKE_BY_KBI,
	WAKE_BY_OTHER
};

u8	CurrentSecond;		// second
u8	CurrentMinute;		// minute
u8	CurrentHour;		// hour
u8	CurrentDay;			// day
u8	CurrentMonth;		// month
u8	CurrentYear;		// year
u8 	DayMax;

code u8 MonthDayTable[]={1,31+1,29+1,31+1,30+1,31+1,30+1,31+1,31+1,30+1,31+1,30+1,31+1};

/*************************************************
*Function: 		char putchar (char c)   
*Description: 	send for printf
*Input:    		char c
*Output:     
*************************************************/
char putchar (char c)   
{      
	bit bES;
	bES=ES0;
    ES0=0;        
    S0BUF = c;        
    while(TI0==0);        
    TI0=0;        
    ES0=bES;        
    return 0;
}


/***********************************************************************************
*Function:   	void INT_SF(void)
*Description:	SF(system flag)Interrupt handler
		 		RTC,WDTF,BOD0F,BOD1F
*Input:   
*Output:     
*************************************************************************************/
void INT_SF(void) interrupt INT_VECTOR_SF
{
	if((PCON1&WDTF)!=0)
	{
		PCON1=WDTF;
	}
	if((PCON1&BOF0)!=0)
	{
		PCON1=BOF0;
	}
	if((PCON1&BOF1)!=0)
	{
		PCON1=BOF1;
	}
	if((PCON1&RTCF)!=0)
	{
		PCON1=RTCF;							// clear interrupt flag
		WakeFlag=WAKE_BY_RTC;
		PCON1 = PCON1 | RTCF;		
		CurrentSecond++;
		if(CurrentSecond>=60)
		{
			CurrentSecond = 0;
			CurrentMinute++;
			if(CurrentMinute>=60)
			{
				CurrentMinute = 0;
				CurrentHour++;
				if(CurrentHour>=24)
				{
					CurrentHour =0;
					CurrentDay++;
					if(CurrentMonth==2)
					{
						if((CurrentYear%4)==0)
						{
							DayMax=29+1;
						}
						else
						{
							DayMax=28+1;
						}
					}
					else
					{
						DayMax=MonthDayTable[CurrentMonth];
					}
					if(CurrentDay>=DayMax)
					{
						CurrentDay=1;
						CurrentMonth++;
						if(CurrentMonth>=13)
						{
							CurrentMonth=1;
							CurrentYear++;
							if(CurrentYear>=100)
							{
								CurrentYear=0;
							}
						}
					}
				}
			}
		}
	}
}


/***********************************************************************************
*Function:   	void INT_KBI(void)	
*Description:	KBI Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_KBI(void)		interrupt INT_VECTOR_KB
{
	INT_DisKBI();				// Disable KBI interrupt
	KBCON&=(~KBIF);				// Clear interrupt flag
	WakeFlag=WAKE_BY_KBI;
}

/***********************************************************************************
*Function:   	void INT_EXINT(void)	
*Description:	INT0 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_INT0(void)		interrupt INT_VECTOR_INT0
{
	INT_DisINT0();				// Disable ExINT0 interrupt
	WakeFlag=WAKE_BY_ExINT0;	
}

/***********************************************************************************
*Function:   	u8 IAP_ReadPPage(u8 PsfrAddr)
*Description:	read P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*Output:     
*		 		u8: sfr data
*************************************************************************************/
u8 IAP_ReadPPage(u8 PsfrAddr)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFMT = ISP_READ_P;			//
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
	return IFD;
}

/***********************************************************************************
*Function:		void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
*Description:	write P page sfr
*Input:   
*		 		u8 PsfrAddr: sfr Address
*		 		u8 PsfrData: sfr data
*Output:     
*************************************************************************************/
void IAP_WritePPage(u8 PsfrAddr,u8 PsfrData)
{
	bit bEA=EA;
	EA = 0; 					//Turn off interrupt
	IFADRH = 0; 				//IFADRH must be 0
	IFADRL= PsfrAddr;			//sfr Address
	IFD= PsfrData;				//sfr data
	IFMT = ISP_WRITE_P;			//
	ISPCR = 0x80;				//Enable ISP/IAP
	SCMD = 0x46;
	SCMD = 0xB9;
	nop();
	IFMT=0;
	ISPCR = 0;					//clear
	EA = bEA;					//restore interrupt
}

/*************************************************
*Function:     	void DelayXus(u8 xUs)
*Description:   	dealy,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    dealy,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitUart0_T1(void)
*Description:   	Initialize Uart0, The source of baud rate was T1
*Input:   
*Output:     
*************************************************************************************/
void InitUart0_T1(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromT1();				// B.R. source:T1
	UART0_SetT1BaudRateX2();			// B.R. x2

	TM_SetT1Mode_2_8BIT_AUTORELOAD();
	TM_SetT1Clock_SYSCLK();
	TM_SetT1Gate_Disable();

    // Sets B.R. value
	TM_SetT1HighByte(T1_BRGRL_9600_2X_12000000_1T);
	TM_SetT1LowByte(T1_BRGRL_9600_2X_12000000_1T);

	TM_EnableT1();
	
}


/***********************************************************************************
*Function:   	void InitRTC(void)
*Description:	Initialize RTC 
*				RTC Clock source: XTAL 32.768KHz
*				RTC Clock prescale: 32768
*				RTC Reload Value: (64-1)=63
*				RTC Overflow freq: 32.768K/32768/1 = 1Hz
*Input:   
*Output:     
*************************************************************************************/
void InitRTC(void)
{	
	u8 x;
	x=IAP_ReadPPage(CKCON2_P);
	x=(x&0x3F)|(ENABLE_XTAL|GAIN_FOR_32K);
	IAP_WritePPage(CKCON2_P,x);
	while(CKCON1 & XTOR == 0x00);		// wait for XTAL stable
	
	RTC_SetClock_ECKI_P60();			// RTC Clock source: XTAL 32.768KHz
	RTC_SetClock_Div_32768();			// RTC Clock prescale: 32768
	RTC_SetReload(64-1);				// RTC Reload Value: (64-1)=63
	RTC_SetCounter(64-1);	
	RTC_ClearRTCF();					// clear RTCF
	RTC_Enable();						// Enable RTC
	
}

/***********************************************************************************
*Function:   	void InitKBI(void)
*Description:   	Initialize KBI
*Input:   
*Output:     
*************************************************************************************/
void InitKBI(void)
{
	KBI_SetKBMask(KBI0|KBI1);				// Enable channel KBI0,KBI1
	KBI_SetKBI01P10P11();					// KBI0,KBI1 set to P10,P11
	KBI_SetKBPattern(KBI0_1|KBI1_1);		// KBI Pattern is 1
	KBI_SetPatternNotEqual();				// Pattern Matching polarity selection : not equal
	KBI_SetFilter_Sysclk_x3();				// KBI Filter: SYSCLK x3
}

/***********************************************************************************
*Function:   	void InitINT0(void)
*Description:   	Initialize INT0
*Input:   
*Output:     
*************************************************************************************/
void InitINT0(void)
{
	INT_SetINT0P30();						// nINT0 : P30
	INT_SetINT0_DetectEdge();				// nINT0 Detect type: edge, On Power-Down mode,only Level.
	INT_SetINT0_DetectLowFalling();			// nINT0 Detect: low level/falling edge
	INT_SetINT0Filter_SysclkDiv6_x3();		// nINT0 filter:(sysclk/6)*3
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnSF_RTC();			// Enable RTC interrupt, must enble SF interrupt
	INT_EnSF();				// Enable SF interrupt
}

/***********************************************************************************
*Function:   	void InitPort(void)
*Description:  	Initialize IO Port
*Input:   
*Output:     
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);					// Set P22,P24,P26 as Push-Pull,For LED.
	PORT_SetP1OpenDrainPu(BIT0|BIT1|BIT5|BIT6);			// Set P10,P11,P15,P16 as open-drain with pull-high
}

/***********************************************************************************
*Function:   	void InitSystem(void)
*Description:   	Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	
	InitPort();
	
	InitRTC();	
	InitKBI();
	InitINT0();
	InitUart0_T1();

	InitInterrupt();		// Initialize Interrupt
	
	INT_EnAll();			// Enable global interrupt

}

/***********************************************************************************
*Function:   	void PowerDown()
*Description:  	Power-Down mode
*Input:   
*Output:     
*************************************************************************************/
void PowerDown(void)
{
	//u8 BakCKCON2;			// Backup CKCON2 (if used PLL CKM)
	u8 i;
	// wait for all of wakup pin P10/P11/P30 to be high
	P10=1;P11=1;P30=1;
	i=0;
	do{
		DelayXus(100);
		if((P10==1)&&(P11==1)&&(P30==1))
		{
			i++;
		}
		else
		{
			i=0;
		}
	}while(i<200);

	// MCK=OSCin
	//BakCKCON2=IAP_ReadPPage(CKCON2_P);							// Backup CKCON2 (if used PLL CKM)
	//IAP_WritePPage(CKCON2_P,BakCKCON2&(~(MCKS0_P|MCKS1_P)));		// MCK=OSCin (if used PLL CKM)

	INT_ClrINT0();		// clear ExINT0 interrupt flag
	INT_ClrKBI();		// clear KBI interrupt flag
	INT_EnINT0();		// Enable ExINT0 interrupt
	INT_EnKBI();		// Enable KBI interrupt
SLEEP_AGAIN:
	WakeFlag=WAKE_BY_OTHER;
	POW_SetMode_PD();			// enter power-down mode

	_nop_();					//					
	_nop_();					//					 

	if(WakeFlag==WAKE_BY_RTC)
	{
		goto SLEEP_AGAIN;
	}
	
	//DelayXus(100);												// delay for stabilize CKM (if used PLL CKM)
	//IAP_WritePPage(CKCON2_P,BakCKCON2);							// restore CKCON2,use CKM (if used PLL CKM)
}


void main()
{
	u8 i;
	
	
    InitSystem();

	CurrentYear=21;
	CurrentMonth=3;
	CurrentDay=9;
	CurrentHour=10;
	CurrentMinute=23;
	CurrentSecond=0;
	
 	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
 	LED_G_0=1;LED_G_1=1;LED_R=1;

					
    while(1)
    {
		printf("\nTime:20%bd-%bd-%bd %bd:%bd:%bd",CurrentYear,CurrentMonth,CurrentDay,CurrentHour,CurrentMinute,CurrentSecond);
    	for(i=0;i<20;i++)
    	{
	    	DelayXms(100);
	    	LED_G_0=!LED_G_0;
    	}
    	// enter sleep
		printf("\nPower down!!");
		PowerDown();
		printf("\nWake up!! 0x%02bX",WakeFlag);
    	
    }
}

