/*********************************************************************
*    Project: MG82F5B32-DEMO
*			MG82F5B32 LQFP32_V10 EV Board (TH156A)
*			CpuCLK=12MHz, SysCLK=12MHz
*    Description:
*			UART0 send and receive data through 485 protocol.
*			P17 control DE/RE
*    Note:
*			
*    Creat time:
*    Modify:
*    
*********************************************************************/
#define _MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>  // for printf

#include ".\include\REG_MG82F5Bxx.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F5B32.H"
#include ".\include\API_Uart_BRGRL_MG82F5B32.H"

/*************************************************
*Set SysClk (MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk (MAX.32MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256) 			
#define TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) /256) 			
#define TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000)))) %256)

#define LED_G_0		P25
#define LED_R		P26
#define LED_G_1		P27

#define IO_485_CTRL		P17

#define Set485Tx()		IO_485_CTRL=1
#define Set485Rx()		IO_485_CTRL=0

#define UART0_RX_BUFF_SIZE   32   		 //Byte
#define UART0_TX_BUFF_SIZE   32   		 //Byte
xdata u8 RcvBuf[UART0_RX_BUFF_SIZE];
u8 Uart0RxIn =0;


u8 UartOvTime;
bit bUartOvFlag;
u8 LedTime;

/***********************************************************************************
*Function:   void INT_UART0(void)
*Description:UART0 Interrupt handler
*		 
*Input:   
*Output:     
*************************************************************************************/
void INT_UART0(void) interrupt INT_VECTOR_UART0
{
	_push_(SFRPI);		   //Store SFRPI value

	SFR_SetPage(0);		   /*Set to page 0*/  //S0CON can only be operated in page 0
	if(TI0)					//Send interrupt
	{
	   TI0 = 0;	   
	}
	if(RI0)					//Receive Interrupt
	{
		RI0 = 0;				//Clear flag
		if(Uart0RxIn<UART0_RX_BUFF_SIZE)
		{
			RcvBuf[Uart0RxIn] = S0BUF;
			Uart0RxIn++;

			UartOvTime=10;
		}
		
	}
	_pop_(SFRPI);		   //restore SFRPI value
}

/***********************************************************************************
*Function:   	void INT_T0(void)
*Description:	T0 Interrupt handler
*Input:   
*Output:     
*************************************************************************************/
void INT_T0(void) interrupt INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;

	if(LedTime!=0) LedTime--;
	if(UartOvTime!=0)
	{
		UartOvTime--;
		if(UartOvTime==0) bUartOvFlag=TRUE;
	}
}

/***********************************************************************************
*Function:		void Uart0SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:     
*************************************************************************************/
void Uart0SendByte(u8 tByte)
{
	bit bES;
	bES=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES;
}

/***********************************************************************************
*Function:		void Uart0SendStr(u8* PStr)
*Description:	send string
*Input: 			u8* PStr:the string to be send
*Output:     
*************************************************************************************/
void Uart0SendStr(u8* PStr)
{
	while(*PStr != 0)
	{
		Uart0SendByte(*PStr);
		PStr ++;
	}
}

/*************************************************
*Function:     	void DelayXus(u16 xUs)
*Description:   	delay,unit:us
*Input:     		u8 Us -> *1us  (1~255)
*Output:     
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if (MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if (MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if (MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif		
#if (MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if (MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:     	void DelayXms(u16 xMs)
*Description:    delay,unit:ms
*Input:     		u16 xMs -> *1ms  (1~65535)
*Output:     
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;
		
	}
}

/***********************************************************************************
*Function:   	void InitPort()
*Description:	Initialize IO Port
*Input:   
*Output:   		
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2OpenDrain(0xFF);				// Set P2 as open drain
	PORT_EnP24P25P26P27PullHigh();			// Enable P24,P25,P26,P27 pull-high
	PORT_SetP3QuasiBi(BIT0|BIT1|BIT3|BIT4|BIT5);		// Set P30,P31,P33,P34,P35 as Quasi-Bidirectional
	PORT_SetP1PushPull(BIT7);							// Set P17 as push pull for 485 control
}

/***********************************************************************************
*Function:   	void InitInterrupt()
*Description:	Initialize Interrupt
*Input:   
*Output:   		
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();			//	Enable T0 interrupt
	INT_EnUART0();			//	Enable UART0 interrupt
	
}	

/***********************************************************************************
*Function:   	void InitUart0_S1BRG()
*Description:	Initialize Uart0, The source of baud rate was S1BRG
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_S1BRG(void)
{
	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	
	UART0_SetBRGFromS1BRG();			// B.R. source: S1BRG
	UART0_SetS1BRGBaudRateX2();			// S1BRG x2
	UART1_SetS1BRGSelSYSCLK();			// S1BRG clock source: Sysclk

    // Sets B.R. value
	UART1_SetS1BRGValue(S1BRG_BRGRL_9600_2X_12000000_1T);
	
	UART1_EnS1BRG();					// Enable S1BRG
}

/***********************************************************************************
*Function:   	void InitUart0_T1()
*Description:	Initialize Uart0, The source of baud rate was T1
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_T1(void)
{
	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	
	UART0_SetBRGFromT1();				// B.R. source: T1
	UART0_SetT1BaudRateX2();			// X2 Baud rate

	TM_SetT1Mode_2_8BIT_AUTORELOAD();
	TM_SetT1Clock_SYSCLK();
	TM_SetT1Gate_Disable();

    // Sets B.R. value
	TM_SetT1HighByte(T1_BRGRL_9600_2X_12000000_1T);
	TM_SetT1LowByte(T1_BRGRL_9600_2X_12000000_1T);

	TM_EnableT1();
}

/***********************************************************************************
*Function:   	void InitUart0_T2()
*Description:	Initialize Uart0, The source of baud rate was T2
*Input:   
*Output:   		
*************************************************************************************/
void InitUart0_T2(void)
{	
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception

	TM_SetT2Mode_BRG();					// B.R. source: T2
	TM_SetT2Clock_SYSCLK();				// TIMER2 Clock source: SYSCLK
	UART0_SetT2BaudRateX1();			// X1

	TM_SetT2RLLowByte(LOBYTE(T2_BRGRL_9600_1X_11059200_1T));
	TM_SetT2RLHighByte(HIBYTE(T2_BRGRL_9600_1X_11059200_1T));
	TM_SetT2LowByte(LOBYTE(T2_BRGRL_9600_1X_11059200_1T));
	TM_SetT2HighByte(HIBYTE(T2_BRGRL_9600_1X_11059200_1T));


	TM_EnableT2();
	
}

/***********************************************************************************
*Function:   	void InitTimer0(void)
*Description:	Initialize Timer0
*Input:   
*Output:   		
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// TIMER0 Mode: 16-bit
	TM_SetT0Clock_SYSCLKDiv12();			// TIMER0 Clock source: SYSCLK/12
	TM_SetT0Gate_Disable();	
				
	TM_SetT0LowByte(TIMER_12T_1ms_TL);		// Set TL0 value
	TM_SetT0HighByte(TIMER_12T_1ms_TH);		// Set TH0 value

	TM_EnableT0();							// Enable TIMER0
}

/***********************************************************************************
*Function:       void InitSystem(void)
*Description:    Initialize MCU
*Input:   
*Output:     
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitTimer0();
	InitUart0_T1();
	//InitUart0_T2();
	//InitUart0_S1BRG();
	InitInterrupt();
}

void main()
{
	u8 i;
	
	InitSystem();

	Uart0RxIn=0;
	UartOvTime=0;
	bUartOvFlag=FALSE;

	INT_EnAll();			//  Enable global interrupt
	
	Set485Rx();
	
	
	LED_G_0=0;LED_G_1=0;LED_R=0;
	DelayXms(1000);
	LED_G_0=1;LED_G_1=1;LED_R=1;	

	Set485Tx();
	DelayXus(10);
	Uart0SendStr("Start!\n");
	DelayXus(10);
	Set485Rx();

	while(1)
	{
		if(LedTime==0)
		{
			LedTime=100;
			LED_G_0 = !LED_G_0;
		}
		if(bUartOvFlag)
		{
			Set485Tx();
			DelayXus(10);
			for(i=0;i<Uart0RxIn;i++)
			{
				Uart0SendByte(RcvBuf[i]);
			}
			Uart0RxIn=0;
			DelayXus(10);
			Set485Rx();
			bUartOvFlag=0;
		}

	}

}


