

/**
 ******************************************************************************
 *
 * @file        main.C
 * @brief       MG32x02z demo main c Code. 
 *
 * @par         Project
 *              MG32x02z
 * @version     V1.00
 * @date        2017/07/07
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 */

#include "MG32x02z_DRV.H"
#include <stdio.h>


typedef uint8_t u8;
typedef uint16_t u16;
typedef uint32_t u32;
typedef uint64_t u64;

void URT0_IRQHandler(void)
{
    uint8_t IRQ_ID;
    
	uint32_t URT_Flag;
	uint32_t URT_IT_Msk;
	
	uint32_t data;
    
    
    IRQ_ID = EXIC_GetITSourceID(URT0_IRQn);      
    if(IRQ_ID & EXIC_SRC5_ID20_urt0_b0)
    {
		
		
		URT_Flag = URT_GetITAllFlagStatus(URT0);
		URT_IT_Msk = URT_GetITStatus(URT0);
		
		if((URT_Flag & URT_STA_RXF_mask_w) && (URT_IT_Msk & URT_INT_RX_IE_mask_w))
		{
			
			//---when RX Data Register have data---//
			data = URT_GetRXData(URT0);
			URT_SetTXData(URT0, 1, data);
			
			// To do ......
			URT_ClearITFlag(URT0,URT_IT_RX);
		}
    }
}

void SysTick_Handler(void)
{
    //to do......
	IncTick();
}


void CSC_Init (void)
{
	CSC_PLL_TyprDef CSC_PLL_CFG;
    
	
    UnProtectModuleReg(MEMprotect);     	// Setting flash wait state
    MEM_SetFlashWaitState(MEM_FWAIT_ONE);	// 50MHz> Sysclk >=25MHz
    ProtectModuleReg(MEMprotect);

    UnProtectModuleReg(CSCprotect);
	CSC_CK_APB_Divider_Select(APB_DIV_1);	// Modify CK_APB divider	APB=CK_MAIN/1
	CSC_CK_AHB_Divider_Select(AHB_DIV_1);	// Modify CK_AHB divider	AHB=APB/1

	
	/* CK_HS selection */
	CSC_IHRCO_Select(IHRCO_12MHz);			// IHRCO Sel 12MHz
	CSC_IHRCO_Cmd(ENABLE);
	while(CSC_GetSingleFlagStatus(CSC_IHRCOF) == DRV_Normal);
	CSC_ClearFlag(CSC_IHRCOF);
	CSC_CK_HS_Select(HS_CK_IHRCO);			// CK_HS select IHRCO


	/* PLL */
	/**********************************************************/
	CSC_PLL_CFG.InputDivider=PLLI_DIV_2;	// 12M/2=6M
	CSC_PLL_CFG.Multiplication=PLLIx16;		// 6M*16=96M
	CSC_PLL_CFG.OutputDivider=PLLO_DIV_2;	// PLLO=96M/2=48M
	CSC_PLL_Config(&CSC_PLL_CFG);
	CSC_PLL_Cmd(ENABLE);
	while(CSC_GetSingleFlagStatus(CSC_PLLF) == DRV_Normal);
	CSC_ClearFlag(CSC_PLLF);
	/**********************************************************/

	
	/* CK_MAIN */ 
	CSC_CK_MAIN_Select(MAIN_CK_PLLO);	


	/* Configure ICKO function */
		
	/* Configure peripheral clock */
	CSC_PeriphProcessClockSource_Config(CSC_UART0_CKS, CK_APB);
 	CSC_PeriphOnModeClock_Config(CSC_ON_UART0,ENABLE);
 	CSC_PeriphOnModeClock_Config(CSC_ON_PortB,ENABLE);
 	CSC_PeriphOnModeClock_Config(CSC_ON_PortC,ENABLE);
 	CSC_PeriphOnModeClock_Config(CSC_ON_PortE,ENABLE);
	
	CSC_PeriphProcessClockSource_Config(CSC_I2C0_CKS, CK_APB);
	CSC_PeriphOnModeClock_Config(CSC_ON_I2C0, ENABLE);					  // Enable IIC0 module clock

 	CSC_PeriphOnModeClock_Config(CSC_ON_DMA,ENABLE);
	

    ProtectModuleReg(CSCprotect);
    
}


int fputc(int ch,FILE *f)
{
	
	URT_SetTXData(URT0,1,ch);
	while(URT_GetITSingleFlagStatus(URT0,URT_IT_TC)==DRV_UnHappened);
	URT_ClearITFlag(URT0,URT_IT_TC);
	
	return ch;
}

void UartSendByte(int ch)
{
	
	URT_SetTXData(URT0,1,ch);
	while(URT_GetITSingleFlagStatus(URT0,URT_IT_TC)==DRV_UnHappened);
	URT_ClearITFlag(URT0,URT_IT_TC);
	
}

void InitGPIO(void)
{
	PIN_InitTypeDef PINX_InitStruct;
    
    
	//==Set GPIO init
	PINX_InitStruct.PINX_Mode				 = PINX_Mode_PushPull_O; 		// Pin select Push Pull mode
	PINX_InitStruct.PINX_PUResistant		 = PINX_PUResistant_Enable;  	// Enable pull up resistor
	PINX_InitStruct.PINX_Speed 			 	 = PINX_Speed_Low;			 
	PINX_InitStruct.PINX_OUTDrive			 = PINX_OUTDrive_Level0;	 	// Pin output driver full strength.
	PINX_InitStruct.PINX_FilterDivider 	 	 = PINX_FilterDivider_Bypass;	// Pin input deglitch filter clock divider bypass
	PINX_InitStruct.PINX_Inverse			 = PINX_Inverse_Disable;	 	// Pin input data not inverse

	PINX_InitStruct.PINX_Alternate_Function  = PE13_AF_GPE13;				// Pin AFS = GPIO
	GPIO_PinMode_Config(PINE(13),&PINX_InitStruct); 					 		// 

	PINX_InitStruct.PINX_Alternate_Function  = PE14_AF_GPE14;				// Pin AFS = GPIO
	GPIO_PinMode_Config(PINE(14),&PINX_InitStruct); 					 		// 

	PINX_InitStruct.PINX_Alternate_Function  = PE15_AF_GPE15;				// Pin AFS = GPIO
	GPIO_PinMode_Config(PINE(15),&PINX_InitStruct); 					 	// 
  



}

void InitUART0(void)
{
    URT_BRG_TypeDef  URT_BRG;
    URT_Data_TypeDef DataDef;
    
	PIN_InitTypeDef PINX_InitStruct;
	//==Set GPIO init
	//PB8 PPO TX ,PB9 ODO RX
	PINX_InitStruct.PINX_Mode				 = PINX_Mode_PushPull_O; 	 	// Pin select Push Pull mode
	PINX_InitStruct.PINX_PUResistant		 = PINX_PUResistant_Enable;  	// Enable pull up resistor
	PINX_InitStruct.PINX_Speed 			 	 = PINX_Speed_Low;			 
	PINX_InitStruct.PINX_OUTDrive			 = PINX_OUTDrive_Level0;	 	// Pin output driver full strength.
	PINX_InitStruct.PINX_FilterDivider 	 	 = PINX_FilterDivider_Bypass;	// Pin input deglitch filter clock divider bypass
	PINX_InitStruct.PINX_Inverse			 = PINX_Inverse_Disable;	 	// Pin input data not inverse
	PINX_InitStruct.PINX_Alternate_Function  = PB8_AF_URT0_TX;				// Pin AFS = URT0_TX
	GPIO_PinMode_Config(PINB(8),&PINX_InitStruct); 					 		// TXD at PB8

	PINX_InitStruct.PINX_Mode				 = PINX_Mode_OpenDrain_O; 		// Pin select Open Drain mode
	PINX_InitStruct.PINX_Alternate_Function  = PB9_AF_URT0_RX;				// Pin AFS = URT0_RX
	GPIO_PinMode_Config(PINB(9),&PINX_InitStruct); 					 		// RXD at PB9

    
    //=====Set Clock=====//
    //---Set BaudRate---//
    URT_BRG.URT_InteranlClockSource = URT_BDClock_PROC;
    URT_BRG.URT_BaudRateMode = URT_BDMode_Separated;
    URT_BRG.URT_PrescalerCounterReload = 0;	                	//Set PSR
    URT_BRG.URT_BaudRateCounterReload = 12;	                	//Set RLR
    URT_BaudRateGenerator_Config(URT0, &URT_BRG);		    	//BR115200 = f(CK_URTx)/(PSR+1)/(RLR+1)/(OS_NUM+1)
    URT_BaudRateGenerator_Cmd(URT0, ENABLE);	            	//Enable BaudRateGenerator
    //---TX/RX Clock---//
    URT_TXClockSource_Select(URT0, URT_TXClock_Internal);		//URT_TX use BaudRateGenerator
    URT_RXClockSource_Select(URT0, URT_RXClock_Internal);		//URT_RX use BaudRateGenerator
    URT_TXOverSamplingSampleNumber_Select(URT0, 31);	        //Set TX OS_NUM
    URT_RXOverSamplingSampleNumber_Select(URT0, 31);	        //Set RX OS_NUM
    URT_RXOverSamplingMode_Select(URT0, URT_RXSMP_3TIME);
    URT_TX_Cmd(URT0, ENABLE);	                            	//Enable TX
    URT_RX_Cmd(URT0, ENABLE);	                            	//Enable RX
    
    

    //=====Set Mode=====//
    //---Set Data character config---//
    DataDef.URT_TX_DataLength  = URT_DataLength_8;
    DataDef.URT_RX_DataLength  = URT_DataLength_8;
    DataDef.URT_TX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_RX_DataOrder   = URT_DataTyped_LSB;
    DataDef.URT_TX_Parity      = URT_Parity_No;
    DataDef.URT_RX_Parity      = URT_Parity_No;
    DataDef.URT_TX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_RX_StopBits    = URT_StopBits_1_0;
    DataDef.URT_TX_DataInverse = DISABLE;
    DataDef.URT_RX_DataInverse = DISABLE;
    URT_DataCharacter_Config(URT0, &DataDef);
    //---Set Mode Select---//
    URT_Mode_Select(URT0, URT_URT_mode);
    //---Set DataLine Select---//
    URT_DataLine_Select(URT0, URT_DataLine_2);
    
    //=====Set Error Control=====//
    // to do...
    
    //=====Set Bus Status Detect Control=====//
    // to do...
    
    //=====Set Data Control=====//
    URT_RXShadowBufferThreshold_Select(URT0, URT_RXTH_1BYTE);
    URT_IdlehandleMode_Select(URT0, URT_IDLEMode_No);
    URT_TXGaudTime_Select(URT0, 0);
    
    //=====Enable URT Interrupt=====//
    //URT_IT_Cmd(URTX, URT_IT_RX, ENABLE);
    //URT_ITEA_Cmd(URTX, ENABLE);
    //NVIC_EnableIRQ(URT0_IRQn);

    //=====Enable URT=====//
    URT_Cmd(URT0, ENABLE);
		
	//==See MG32x02z_URT0_IRQ.c when interrupt in
}


DRV_Return I2C_DeInit(I2C_Struct* I2Cx)
{
    if(__I2C_GetFlagStatus(I2Cx, I2C_FLAG_BUSYF))
        return DRV_Busy;

    __I2C_Disable(I2Cx);

    I2Cx->CR0.W = 0;
    I2Cx->STA.W = (~(I2C_FLAG_EVENTF | I2C_FLAG_TXF));
    I2Cx->INT.W = 0;
    I2Cx->CLK.W = 0;
    I2Cx->CR1.W = 0;
    I2Cx->CR2.W = 0;
    I2Cx->SADR.W = 0;
    I2Cx->TMOUT.W = 0;
    I2Cx->CR0.W = 0;

    return DRV_Success;
}


void InitI2C0(void)
{
	uint32_t			CK_I2C_PR_Frequency;
	uint32_t			SCL_Clock;

    uint16_t lI2C_Pre = 1;
    uint16_t lI2C_DIV = 1;
    uint16_t lI2C_DIV_INX = 1;
    uint16_t lI2C_HT_LT = 0;
    uint16_t lI2C_LT;
    uint16_t lI2C_HT;
    
	PIN_InitTypeDef PINX_InitStruct;

    PINX_InitStruct.PINX_Mode	            = PINX_Mode_OpenDrain_O;        // Pin select Open Drain mode
    PINX_InitStruct.PINX_PUResistant        = PINX_PUResistant_Enable;      // Enable pull up resistor
    PINX_InitStruct.PINX_Speed              = PINX_Speed_High;           
    PINX_InitStruct.PINX_OUTDrive           = PINX_OUTDrive_Level0;         // Pin output driver full strength.
    PINX_InitStruct.PINX_FilterDivider      = PINX_FilterDivider_Bypass;    // Pin input deglitch filter clock divider bypass
    PINX_InitStruct.PINX_Inverse            = PINX_Inverse_Disable;         // Pin input data not inverse

    PINX_InitStruct.PINX_Alternate_Function = PB10_AF_I2C0_SCL;              // Pin AFS = I2C0_SCL 
    GPIO_PinMode_Config(PINB(10),&PINX_InitStruct);                         
    
    PINX_InitStruct.PINX_Alternate_Function = PB11_AF_I2C0_SDA;              // Pin AFS = I2C0_SDA
    GPIO_PinMode_Config(PINB(11),&PINX_InitStruct);                         

	__I2C_Disable(I2C0);
	I2C_DeInit(I2C0);
    CK_I2C_PR_Frequency = 48000000;
    SCL_Clock = 400000;    

    //=== I2C Config ===//
    //===== I2C Output Clock Config =====//
    // CK_I2C_PR
    // SCL Output Clock
    // HT + LT, <= 32 >=9, CK_I2C_PR / SCL Clock / Prescaler / DIV

    do{
        lI2C_HT_LT = CK_I2C_PR_Frequency / SCL_Clock / lI2C_Pre / lI2C_DIV;
        if((lI2C_HT_LT >= 32) || (lI2C_HT_LT <=9)) 
        {
            lI2C_Pre ++;
            if(lI2C_Pre > 8)
            {
                lI2C_Pre = 1;
                lI2C_DIV =lI2C_DIV*2;
                lI2C_DIV_INX++;
            }
        }
    }while((lI2C_HT_LT >= 32) || (lI2C_HT_LT <=9));

    lI2C_LT = (lI2C_HT_LT >> 1);
    lI2C_HT = lI2C_HT_LT - lI2C_LT;

    __I2C_SetClockSource(I2C0, I2C_CLK_SRC_PROC);
    __I2C_SetClockPrescaler(I2C0, lI2C_Pre - 1);
    __I2C_SetClockDivider(I2C0, lI2C_DIV_INX - 1);
    __I2C_SetSCLHighTime(I2C0, lI2C_HT - 1);
    __I2C_SetSCLLowTime(I2C0, lI2C_LT - 1);

	//===== I2C Opration Mode Config =====//
	__I2C_GeneralCallAddress_Disable(I2C0);
	__I2C_SlaveAddressDetect_Disable(I2C0, (I2C_SADR_1 | I2C_SADR_2));
		
	//===== I2C Interrupt Config =====//
	//		  NVIC_EnableIRQ(I2C0_IRQn);
	//		  NVIC_SetPriority(I2C0_IRQn, 1);										  // Suggest SYSTICK Priority = 0
	//																				  //		   Other Priority > 0
	
	__I2C_ITEA_Disable(I2C0);
	__I2C_IT_Disable(I2C0, (I2C_IT_TMOUT | I2C_IT_EVENT | I2C_IT_ERR | I2C_IT_BUF | I2C_IT_WUP));

	//===== I2C Timeout Config =====//
	__I2C_TMO_Enable(I2C0);
	__I2C_SetTimeOutClockSource(I2C0, I2C_TMO_CKS_DIV64);
	__I2C_SetTimeOutDetectionMode(I2C0, I2C_TMO_MDS_GENERAL);
	__I2C_SetTimeOutCount(I2C0, I2C_TMO_MDS_GENERAL);

	//===== I2C Enable =====//
	__I2C_Enable(I2C0);
}

u8 WaitI2C0HandleDone(void)
{
	u8 lState;
	__I2C_SetTimeOutCount(I2C0, 100);
    __I2C_ClearFlag(I2C0,I2C_FLAG_ALL);
    while(__I2C_GetStateFlag(I2C0) == 0)
    {
		if(I2C_GetFlagStatus(I2C0,I2C_FLAG_TMOUTF)==DRV_Happened)
		{
			return DRV_Failure;
		}
    }
    lState = __I2C_GetStatusCode(I2C0);
    if((lState==0x00)||(lState>=0xF8))
    {
		return DRV_Failure;
    }
    
    return DRV_Success;
}

u8 I2CRead(u8 DevAddr,u8 RegAddr,u8 *pBuf,u8 Len)
{
	u8 i;

	Set_STA_STO_AA_100(I2C0);	// START
	
	if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ
	Set_STA_STO_AA_000(I2C0);	// ַ
	__I2C_SendSBUF(I2C0, DevAddr&0xFE);
	if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ

	Set_STA_STO_AA_001(I2C0);	// ͵ַ
	__I2C_SendSBUF(I2C0, RegAddr);
	if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ
	Set_STA_STO_AA_100(I2C0);	// ٴηSTART
	if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ

	Set_STA_STO_AA_000(I2C0);	// ַ ,ģʽ
	__I2C_SendSBUF(I2C0, DevAddr|0x01);
	if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ
	i=0;
	while(i<Len)
	{
		i++;
		if(i==Len)
			Set_STA_STO_AA_000(I2C0);
		else
			Set_STA_STO_AA_001(I2C0);
		if(WaitI2C0HandleDone()==DRV_Failure) return DRV_Failure;		// ȴ
		pBuf[i-1]=__I2C_ReceiveSBUF(I2C0);
	}

	Set_STA_STO_AA_010(I2C0);	// STOP
    __I2C_ClearStateFlag(I2C0);
	
	return DRV_Success;
}

int main()
{
	u8 i=0;
	u8 x[10];
	CSC_Init();
	InitGPIO();
	InitUART0();

	InitI2C0();	

	
	InitTick(48000000,0);
	NVIC_EnableIRQ(SysTick_IRQn);

	printf("Hello!\n");

	i=0;
	while(1)
    {
		Delay(100);
		PE15=!PE15;
		if(I2CRead(0xA0,i,x,10)==DRV_Failure) 
		{
			PE14=0;
			InitI2C0();
			PE14=1;
		}
		printf("\nI2C0:%x,%x ",i,x[0]);
		i++;
		if(i==x[0])
		{
			PE13=0;
			printf("PASS");
			PE13=1;
		}
		else
		{
			printf("FAIL");
		}
		
    }
}




















